import asyncio
import logging
import os
import pwd
import random
import re
import subprocess
import time
import aiohttp
import grp
from urllib.request import urlopen, Request, HTTPError, URLError

# For compatibility with python versions 3.6 or earlier.
# asyncio.Task.all_tasks() is fully moved to asyncio.all_tasks() starting with 3.9
try:
  asyncio_all_tasks = asyncio.all_tasks
except AttributeError as e:
  asyncio_all_tasks = asyncio.Task.all_tasks

def random_suffix():
  return "." + str(int(time.time())) + "-" + str(random.randrange(100000, 999999))

def backtick(command):
  p = subprocess.Popen(command, stdout=subprocess.PIPE)
  output = p.stdout.read().decode()
  p.wait()

  if p.returncode != 0:
    raise Exception("Command " + str(command) + " failed: " + str(p.returncode))

  return output

def getuidgid(username, groupname = None):
  pwent = pwd.getpwnam(username)
  uid = pwent.pw_uid
  gid = pwent.pw_gid

  if groupname is not None:
    groupinfo = grp.getgrnam(groupname)
    gid = groupinfo.gr_gid

  return (uid, gid)

def sanitizedLog(string):
  string = re.sub(r"(.*?password=).*? ", r"\1********** ", string)
  return re.sub(r"(-u\s+['\"][^:]+:)([^'\"]+)(['\"])", r"\1*****\3", string)

def replace_env_tokens(data, config=None):
  if data is None:
    return None
  elif isinstance(data, str):
    return _replace_env_tokens(data, config)
  return [_replace_env_tokens(x, config) for x in data]

async def subprocess_call(command, wait_time=60, retry=False, **kwds):
  """
  Retry a command potentially a few times, with a timeout given by wait_time.
  If [command] is a string, it will execute using create_subprocess_shell(), else it will use create_subprocess_exec().
  """
  logger = logging.getLogger(__name__)
  start_time = time.time()

  while True:
    if isinstance(command, str):
      process = await asyncio.create_subprocess_shell(command, stdout=asyncio.subprocess.PIPE,
                                                      stderr=asyncio.subprocess.STDOUT, **kwds)
    else:
      process = await asyncio.create_subprocess_exec(*command, stdout=asyncio.subprocess.PIPE,
                                                     stderr=asyncio.subprocess.STDOUT, **kwds)

    stdout, stderr = await asyncio.wait_for(process.communicate(), wait_time)
    output = stdout.decode().strip()

    if process.returncode == 0:
      logger.info("<OK> %s%s", sanitizedLog(str(command)), " [out -> {}]".format(output) if output else "")
      return output
    elif not retry:
      raise subprocess.CalledProcessError(process.returncode, command, output=output)
    elif time.time() > start_time + wait_time:
      logger.warning("<Retries exhausted> %s%s", sanitizedLog(str(command)), " [out -> {}]".format(output) if output else "")
      raise subprocess.CalledProcessError(process.returncode, command, output=output)
    else:
      logger.info("<Retrying> %s%s", sanitizedLog(str(command)), " [out -> {}]".format(output) if output else "")
      await asyncio.sleep(2)

# Borrowed from Python 3.7 provisional code
def asyncio_run(main, *, debug=False):
  """Run a coroutine.
  This function runs the passed coroutine, taking care of
  managing the asyncio event loop and finalizing asynchronous
  generators.
  This function cannot be called when another asyncio event loop is
  running in the same thread.
  If debug is True, the event loop will be run in debug mode.
  This function always creates a new event loop and closes it at the end.
  It should be used as a main entry point for asyncio programs, and should
  ideally only be called once.
  Example:
      async def main():
          await asyncio.sleep(1)
          print('hello')
      asyncio.run(main())
  """
  if asyncio.events._get_running_loop() is not None:
    raise RuntimeError(
      "asyncio_run() cannot be called from a running event loop")

  if not asyncio.coroutines.iscoroutine(main):
    raise ValueError("a coroutine was expected, got {!r}".format(main))

  loop = asyncio.events.new_event_loop()
  try:
    asyncio.events.set_event_loop(loop)
    loop.set_debug(debug)
    return loop.run_until_complete(main)
  finally:
    try:
      _cancel_all_tasks(loop)
      loop.run_until_complete(loop.shutdown_asyncgens())
    finally:
      asyncio.events.set_event_loop(None)
      loop.close()

def _cancel_all_tasks(loop):
  to_cancel = asyncio_all_tasks(loop)
  if not to_cancel:
    return

  for task in to_cancel:
    task.cancel()

  loop.run_until_complete(
    asyncio.tasks.gather(*to_cancel, loop=loop, return_exceptions=True))

  for task in to_cancel:
    if task.cancelled():
      continue
    if task.exception() is not None:
      loop.call_exception_handler({
        'message': 'unhandled exception during asyncio.run() shutdown',
        'exception': task.exception(),
        'task': task,
      })

def _replace_env_tokens(string, config):
  for token in set(re.findall(r'@{([A-Z_0-9]+)}', string)):
    val = os.environ.get(token)

    if val is None and config is not None:
      val = config.get(token.lower())

    if val is not None:
      string = string.replace("@{{{}}}".format(token), str(val))
  return string

async def check_http_get_ok(url, ssl_context=None, timeout=5*60):
  """Check HTTP GET request response is ok

  url - full url of the HTTP endpoint to check
  timeout - time in seconds for the entire operation to complete
  """
  timeout = aiohttp.ClientTimeout(total=timeout)
  async with aiohttp.ClientSession(timeout=timeout) as session:
    async with session.get(url, ssl=ssl_context) as resp:
      if resp.status / 100 != 2:
        raise RuntimeError(
          f"HTTP endpoint [{url}] returned non 2xx status [{resp.status}]"
        )

class ThrottledManagerNotifier:
  """Manager notifier that throttles notifications

  url - full url of the HTTP endpoint to notify on the manager
  wait_sec - time in seconds between throttled calls
  """

  def __init__(self, url, wait_sec):
    self.url = url
    self.wait_sec = wait_sec
    self.task = None
    self.log = logging.getLogger(__name__)

  def __call_manager(self):
    try:
      response = urlopen(Request(self.url, method="POST"))
      if response.status / 100 != 2:
        self.log.debug(f"Unable to notify manager, HTTP endpoint [{self.url}] returned non 2xx status [{response.status}]")
    except (HTTPError, URLError, ValueError):
        self.log.exception(f"Unable to notify manager, HTTP endpoint [{self.url}] ran into exception")

  async def __call(self):
    await asyncio.sleep(self.wait_sec)
    self.__call_manager()
    self.task = None

  def maybe_notify(self):
    """Maybe notifies manager
    noop if url is None or empty or ongoing task
    """
    if not self.url:
      return

    if self.task is not None:
      return

    self.task = asyncio.ensure_future(self.__call())
