import logging

try:
  import psutil
except ModuleNotFoundError:
  psutil = None

def get_resources():
  logger = logging.getLogger(__name__)

  if not psutil:
    logger.debug("Could not get system resources because psutil was not loaded")
    return {}

  return {"cpu_count": psutil.cpu_count(),
          "total_mem_gb": round(psutil.virtual_memory().total / (1024 * 1024 * 1024), 1),
          "disk_partitions": get_disk_partitions()}

def get_disk_partitions():
  logger = logging.getLogger(__name__)

  if not psutil:
    logger.debug("Could not get disk partitions because psutil was not loaded")
    return []

  partitions = []
  for partition in psutil.disk_partitions():
    partitions.append({"device": partition.device,
                       "mount_point": partition.mountpoint,
                       "total_gb": round(psutil.disk_usage(partition.mountpoint).total / (1024 * 1024 * 1024), 1)})

  return partitions

def get_system_stats():
  logger = logging.getLogger(__name__)

  if not psutil:
    logger.debug("Could not get system stats because psutil was not loaded")
    return {}

  cpu = psutil.cpu_times_percent(percpu=True)
  cpus = [{"id": i, "user": x.user, "nice": x.nice, "sys": x.system, "idle": x.idle} for i, x in enumerate(cpu)]

  mem = psutil.virtual_memory()
  mem_dict = {"total": mem.total, "available": mem.available, "used": mem.used, "free": mem.free,
              "percent": mem.percent}
  if hasattr(mem, 'buffers'):
    mem_dict["buffers"] = mem.buffers
  if hasattr(mem, 'cached'):
    mem_dict["cached"] = mem.cached

  swap_mem = psutil.swap_memory()
  swap_mem_dict = {"total": swap_mem.total, "used": swap_mem.used, "free": swap_mem.free, "percent": swap_mem.percent}

  partitions = []
  for partition in psutil.disk_partitions():
    disk_usage = psutil.disk_usage(partition.mountpoint)
    partitions.append(
      {"device": partition.device, "mount": partition.mountpoint, "total": disk_usage.total, "used": disk_usage.used,
       "free": disk_usage.free, "percent": disk_usage.percent})

  return {"cpu": cpus,
          "mem": mem_dict,
          "swap_mem": swap_mem_dict,
          "disk": partitions
          }
