import hashlib
import json
import logging
import os
import re
import ssl
import time

from . import dirs
from . import fetch
from . import fs
from . import util

def setup(config):
  user = config["user"]
  group = config.get("group", None)
  homedir = config["homedir"]
  vardir = dirs.vardir(homedir)

  # Create vardir
  if not os.path.exists(vardir):
    (uid, gid) = util.getuidgid(user, group)
    fs.mkdirs(vardir)
    os.chown(vardir, uid, gid)

async def pull(config, service_yaml, service_type):
  """Fetches all files needed to become a service.
  Does not actually activate or become that service. Returns the directory that the deployment was pulled to."""

  logger = logging.getLogger(__name__)

  hashed_service_parts = hash_service_parts(service_yaml, service_type)
  service_directory = make_deploy_directory(config, hashed_service_parts)
  if os.path.exists(service_directory):
    logger.info("Service directory [%s] for hash [%s] already exists, skipping pull", service_directory,
                hashed_service_parts)
    return service_directory

  # If provided, server_ca_cert is applied to all HTTP repositories.
  # Applying certificates per HTTP repository is not supported
  ssl_context = None
  if config.get("server_ca_cert") is not None:
    ssl_context = ssl.create_default_context(ssl.Purpose.SERVER_AUTH)
    ssl_context.load_verify_locations(config.get("server_ca_cert"))

  logger.info("Pulling deployment bundles: [{}]".format(', '.join(service_yaml["deploy"])))

  last_exception = None
  for attempt in range(3):
    # Prepare in a temporary directory, then rename to the real directory
    suffix = util.random_suffix()
    tmp_service_directory = service_directory + suffix
    fs.mkdirs(tmp_service_directory)

    try:
      tarball_paths = await fetch.fetch_files_async(["{}.tar.gz".format(x) for x in service_yaml["deploy"]],
                                                    dirs.tgzdir(config["homedir"]),
                                                    fetch.get_repository_list(config),
                                                    attempt > 1,
                                                    ssl_context=ssl_context)

      # Populate temporary directory
      for tarball_path in tarball_paths:
        logger.info("Extracting [%s]", tarball_path)
        await util.subprocess_call(
          ["tar", "--no-same-owner", "--keep-old-files", "-C", tmp_service_directory, "-xf", tarball_path])

      # Fill in templated configs
      conf_dir = os.path.join(tmp_service_directory, "conf")
      conf_env = build_environment(service_yaml, service_type)

      logger.info("Filling in templated configs at [%s]", conf_dir)
      expand_config_templates(conf_dir, conf_env)

      # Write deploy details to deploy.json
      deploy_file = os.path.join(tmp_service_directory, "deploy.json")

      logger.info("Writing deploy details to [%s]", deploy_file)
      fs.write_file(deploy_file, json.dumps(combine_service_parts(service_yaml, service_type)) + "\n", 0o644)

      # Rename temporary directory -> real directory
      os.rename(tmp_service_directory, service_directory)

      return service_directory
    except Exception as e:
      fs.rmr(tmp_service_directory)

      if os.path.exists(service_directory):
        # If it *did* exist, someone beat us to creating it, and that's okay.
        return service_directory

      logger.info("Failed to pull and prepare deployment, retrying: %s", e)
      last_exception = e

  else:
    logger.info("Retries exhausted while pulling and preparing deployment")
    raise last_exception

def is_activated(homedir, deploy_directory):
  """Checks if a particular deploy_directory is the target of the deploy symlink."""
  deploy_link = dirs.deploydir(homedir)
  return os.path.lexists(deploy_link) and os.readlink(deploy_link) == deploy_directory

def activate(homedir, deploy_directory):
  """Activates a particular deploy_directory by updating the deploy symlink."""
  deploy_link = dirs.deploydir(homedir)
  if not os.path.lexists(deploy_link):
    os.symlink(deploy_directory, deploy_link)
  elif not os.path.islink(deploy_link):
    raise Exception("deploydir is not a symlink: " + deploy_link)
  elif not os.readlink(deploy_link) == deploy_directory:
    os.unlink(deploy_link)
    os.symlink(deploy_directory, deploy_link)

def activated_hash(homedir):
  """Returns the hash (like the one from hash_service_parts) of the currently activated directory, or None."""
  deploy_link = dirs.deploydir(homedir)
  if os.path.lexists(deploy_link):
    return os.path.basename(os.readlink(deploy_link))
  else:
    return None

def read_deploy_json(deployjson_file):
  deployjson = json.loads(fs.read_file(deployjson_file))
  return (deployjson["yaml"], deployjson["type"])

def build_environment(service_yaml, service_type):
  """Given service_yaml and service_type, returns a dict representing the runtime environment."""
  env = {}
  for var, value in service_yaml["env"].items():
    if isinstance(value, dict):
      # Find most specific match for our type
      match_len = 0
      match_str = None
      for type_prefix, v in value.items():
        if (type_prefix == service_type or service_type.startswith(type_prefix + "/")) and len(type_prefix) > match_len:
          match_len = len(type_prefix)
          match_str = v
      if match_str is not None:
        env[var] = str(match_str)
      else:
        raise Exception("no value for env var: " + var)
    else:
      env[var] = str(value)
  return env

def combine_service_parts(service_yaml, service_type):
  """Given service_yaml and service_type, returns a dict that should become deploy.json."""
  return {"name": service_yaml["name"], "yaml": service_yaml, "type": service_type}

def hash_service_parts(service_yaml, service_type):
  """Given service_yaml and service_type, returns a hash we can use in their stead."""
  canonical_json = json.dumps(combine_service_parts(service_yaml, service_type), sort_keys=True)
  return hashlib.sha1(canonical_json.encode()).hexdigest()[:12]

def make_deploy_directory(config, hashed_service_parts):
  """Given the output of hash_service_parts, returns the directory we should deploy to."""
  parent_directory = dirs.deploystagedir(config["homedir"])
  return os.path.join(parent_directory, hashed_service_parts)

def expand_config_templates(conf_dir, conf_env):
  """Walk through conf_dir, using conf_env to expand any @{XXX} variables along the way."""
  if os.path.exists(conf_dir) and os.path.isdir(conf_dir):
    for conf_file in os.listdir(conf_dir):
      conf_path = os.path.join(conf_dir, conf_file)
      if os.path.islink(conf_path):
        pass
      elif os.path.isfile(conf_path):
        try:
          conf_contents = fs.read_file(conf_path)
          new_conf_contents = re.sub(
            r'\@\{([\w-]+)(|\|[^\}]*)\}',
            lambda m: conf_env[m.group(1)] if m.group(2) == '' else conf_env.get(m.group(1), m.group(2)[1:]),
            conf_contents
          )
          if new_conf_contents != conf_contents:
            fs.write_file(conf_path, new_conf_contents)
        except UnicodeDecodeError:
          # file is not a unicode text file, skip replacing
          pass
      elif os.path.isdir(conf_path):
        expand_config_templates(conf_path, conf_env)

def get_keep_history(config):
  # Keep 10 directories by default
  return config.get("keep_history", 10)

def read_history(config):
  """Returns a dict representing the contents of the history file."""
  homedir = config["homedir"]
  keep_history = get_keep_history(config)
  history_file = dirs.historyjson(homedir)
  if os.path.exists(history_file):
    return dict(json.loads(fs.read_file(history_file)))
  else:
    return {"history": []}

def append_history_and_prune_old_files(config, deploy_directory):
  """Update the history file, noting that deploy_directory was deployed.

  Also prunes older histories, up to keep_history. Should only be called while the homedir is locked, otherwise
  havoc can ensue."""
  homedir = config["homedir"]
  keep_history = max(1, get_keep_history(config))
  history_file = dirs.historyjson(homedir)
  history_json = read_history(config)
  hashed_service_parts = os.path.basename(deploy_directory)

  # Append, prune history_json
  new_history_entry = {"timestamp": time.time(), "hashed_service_parts": hashed_service_parts}
  history_json["history"].append(new_history_entry)
  while len(history_json["history"]) > keep_history:
    history_json["history"].pop(0)

  # Write new history file
  try:
    tmp_history_file = history_file + util.random_suffix()
    fs.write_file(tmp_history_file, json.dumps(history_json) + "\n", 0o644)
    os.rename(tmp_history_file, history_file)
  except Exception as e:
    fs.rmr(tmp_history_file)
    raise

  # Decide which deploy directories and tgzs to keep
  keep_dirs = {}
  keep_tgzs = {}
  for history_entry in history_json["history"]:
    hashed_service_parts = history_entry["hashed_service_parts"]
    deploy_directory = make_deploy_directory(config, hashed_service_parts)
    keep_dirs[hashed_service_parts] = 1

    if (os.path.isdir(deploy_directory)):
      service_yaml, service_type = read_deploy_json(os.path.join(deploy_directory, "deploy.json"))
      for tgz in service_yaml["deploy"]:
        keep_tgzs[tgz + ".tar.gz"] = 1

  # Prune old deploy directories
  for hashed_service_parts in os.listdir(dirs.deploystagedir(homedir)):
    deploy_directory = make_deploy_directory(config, hashed_service_parts)
    if os.path.isdir(deploy_directory) and hashed_service_parts not in keep_dirs and not is_activated(homedir,
                                                                                                      deploy_directory):
      fs.rmr(deploy_directory)

  # Prune old tgzs
  if os.path.exists(dirs.tgzdir(homedir)):
    for tgz_name in os.listdir(dirs.tgzdir(homedir)):
      tgz_path = os.path.join(dirs.tgzdir(homedir), tgz_name)
      if os.path.isfile(tgz_path) and tgz_name not in keep_tgzs:
        os.unlink(tgz_path)
