/*
 * Decompiled with CFR 0.152.
 */
package io.imply.telemetry;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.imply.telemetry.CheckedSupplier;
import io.imply.telemetry.ClarityEmitterConfig;
import io.imply.telemetry.Emitter;
import io.imply.telemetry.EmitterException;
import io.imply.telemetry.Event;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClarityEmitter
implements Emitter {
    private static final Logger log = LoggerFactory.getLogger(ClarityEmitter.class);
    private static final AtomicInteger INSTANCE_COUNTER = new AtomicInteger();
    private static final long BUFFER_FULL_WARNING_THROTTLE = 30000L;
    private static final Random RANDOM = new SecureRandom();
    private static final long CONSECUTIVE_TOO_MANY_REQUESTS_WARN = 12L;
    static final int MAX_EVENT_SIZE = 1047552;
    private final ObjectMapper jsonMapper = new ObjectMapper();
    private final ScheduledExecutorService exec;
    private final LinkedBlockingQueue<byte[]> queue = new LinkedBlockingQueue();
    private final AtomicLong bufferedSize = new AtomicLong(0L);
    private final Object sizeLock = new Object();
    private final AtomicInteger droppedEvents = new AtomicInteger(0);
    private final long bufferedBytesForFlush;
    private final AtomicLong lastBufferFullWarning = new AtomicLong(0L);
    private final AtomicInteger emittingRunnableRefCount = new AtomicInteger(0);
    private final AtomicLong consecutiveTooManyRequests = new AtomicLong(0L);
    private final ClarityEmitterConfig config;
    private final String serviceName;
    private final String credentials;
    private final CheckedSupplier<URLConnection, IOException> connectionSupplier;

    public ClarityEmitter(ClarityEmitterConfig config, String serviceName) {
        this(Objects.requireNonNull(config, "config cannot be null").validate(), serviceName, config.baseUrl::openConnection, () -> Executors.newSingleThreadScheduledExecutor(new EmitterThreadFactory()));
    }

    ClarityEmitter(ClarityEmitterConfig config, String serviceName, CheckedSupplier<URLConnection, IOException> connectionSupplier, Supplier<ScheduledExecutorService> executorSupplier) {
        this.config = config;
        this.serviceName = Objects.requireNonNull(serviceName, "serviceName cannot be null");
        this.connectionSupplier = connectionSupplier;
        this.exec = executorSupplier.get();
        this.bufferedBytesForFlush = (long)((double)(config.maxBufferSize * (long)config.flushBufferPercentFull) / 100.0);
        this.credentials = Base64.getEncoder().encodeToString(String.format("%s:%s", config.username, config.password).getBytes(StandardCharsets.UTF_8));
        long period = (long)((double)config.flushMillis + RANDOM.nextDouble() * 0.4 + 0.8);
        this.exec.scheduleWithFixedDelay(new ScheduledEmittingRunnable(), period, period, TimeUnit.MILLISECONDS);
    }

    @Override
    public void emit(Event event) {
        if (this.exec.isShutdown()) {
            throw new RejectedExecutionException("Emitter is closed");
        }
        Optional<byte[]> data = this.encodeEvent(Objects.requireNonNull(event, "Event cannot be null"));
        if (!data.isPresent()) {
            return;
        }
        if (!this.enqueue(data.get())) {
            this.droppedEvents.incrementAndGet();
        }
        if (this.config.flushCount > 0 && this.queue.size() >= this.config.flushCount && this.emittingRunnableRefCount.compareAndSet(0, 1)) {
            log.info("Buffer flushCount [{} >= {}] exceeded, flushing.", (Object)this.queue.size(), (Object)this.config.flushCount);
            this.exec.submit(new EmittingRunnable());
        } else if (this.bufferedSize.get() >= this.bufferedBytesForFlush && this.emittingRunnableRefCount.compareAndSet(0, 1)) {
            if (log.isInfoEnabled()) {
                log.info("Buffer flushBufferPercentFull [{} >= {}] exceeded, flushing.", (Object)new DecimalFormat("#.##").format((double)this.bufferedSize.get() / (double)this.config.maxBufferSize * 100.0), (Object)this.config.flushBufferPercentFull);
            }
            this.exec.submit(new EmittingRunnable());
        }
        this.lastBufferFullWarning.getAndUpdate(lastLog -> {
            long now = System.currentTimeMillis();
            if (this.droppedEvents.get() > 0 && lastLog + 30000L < now) {
                log.error("Buffer full: dropped {} events!", (Object)this.droppedEvents.get());
                this.droppedEvents.set(0);
                return now;
            }
            return lastLog;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean enqueue(byte[] event) {
        Object object = this.sizeLock;
        synchronized (object) {
            if (this.bufferedSize.get() + (long)event.length <= this.config.maxBufferSize && this.queue.offer(event)) {
                this.bufferedSize.addAndGet(event.length);
                return true;
            }
        }
        return false;
    }

    private Optional<byte[]> encodeEvent(Event event) {
        byte[] eventBytes;
        HashMap<String, Object> map = new HashMap<String, Object>(this.config.attributes);
        map.putAll(event.toMap());
        map.put("service", this.serviceName);
        map.put("feed", event.getFeed());
        try {
            eventBytes = this.jsonMapper.writeValueAsBytes(map);
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
        if (eventBytes.length > 1047552 && log.isErrorEnabled()) {
            log.error("Event too large to emit ({} > {}): {} ...", new Object[]{eventBytes.length, 1047552, new String(Arrays.copyOf(eventBytes, 1024), StandardCharsets.UTF_8)});
            return Optional.empty();
        }
        return Optional.of(eventBytes);
    }

    @Override
    public void flush() throws IOException {
        try {
            Future<?> future = this.exec.submit(new EmittingRunnable());
            this.emittingRunnableRefCount.incrementAndGet();
            future.get(this.config.flushTimeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Thread interrupted while flushing", e);
        }
        catch (ExecutionException e) {
            throw new IOException("Exception while flushing", e);
        }
        catch (TimeoutException e) {
            throw new IOException(String.format("Timed out after [%d] millis during flushing", this.config.flushTimeout), e);
        }
        catch (RejectedExecutionException rejectedExecutionException) {
            // empty catch block
        }
    }

    @Override
    public void close() throws IOException {
        if (this.exec.isShutdown()) {
            return;
        }
        Future<?> future = this.exec.submit(new EmittingRunnable());
        this.exec.shutdown();
        try {
            future.get(this.config.flushTimeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Thread interrupted while closing", e);
        }
        catch (ExecutionException e) {
            throw new IOException("Exception while closing", e);
        }
        catch (TimeoutException e) {
            throw new IOException(String.format("Timed out after [%d] millis during closing", this.config.flushTimeout), e);
        }
    }

    private static class EmitterThreadFactory
    implements ThreadFactory {
        private EmitterThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r, String.format("ImplyTelemetryEmitter-%d", INSTANCE_COUNTER.incrementAndGet()));
            thread.setDaemon(true);
            return thread;
        }
    }

    private class ScheduledEmittingRunnable
    extends EmittingRunnable {
        private ScheduledEmittingRunnable() {
        }

        @Override
        public void run() {
            ClarityEmitter.this.emittingRunnableRefCount.incrementAndGet();
            super.run();
        }
    }

    private class EmittingRunnable
    implements Runnable {
        private EmittingRunnable() {
        }

        @Override
        public void run() {
            try {
                this.emit();
            }
            finally {
                ClarityEmitter.this.emittingRunnableRefCount.decrementAndGet();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void emit() {
            LinkedList<byte[]> taken = new LinkedList<byte[]>();
            Object object = ClarityEmitter.this.sizeLock;
            synchronized (object) {
                ClarityEmitter.this.queue.drainTo(taken);
                ClarityEmitter.this.bufferedSize.set(0L);
            }
            Optional<LinkedList<LinkedList<byte[]>>> batches = this.splitIntoBatches(taken);
            if (!batches.isPresent()) {
                return;
            }
            block7: for (LinkedList linkedList : batches.get()) {
                while (true) {
                    try {
                        if (this.sendBatch(linkedList)) {
                            continue block7;
                        }
                    }
                    catch (Exception e) {
                        log.warn("Got exception when posting events to [{}], retrying.", (Object)((ClarityEmitter)ClarityEmitter.this).config.baseUrl, (Object)e);
                    }
                    try {
                        Thread.sleep(100L);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        return;
                    }
                }
            }
        }

        private boolean sendBatch(LinkedList<byte[]> batch) throws IOException {
            int code;
            String output;
            HttpURLConnection connection = (HttpURLConnection)ClarityEmitter.this.connectionSupplier.get();
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("Authorization", String.format("Basic %s", ClarityEmitter.this.credentials));
            if (ClarityEmitterConfig.Compression.GZIP.equals((Object)((ClarityEmitter)ClarityEmitter.this).config.compression)) {
                connection.setRequestProperty("Content-Encoding", "gzip");
            }
            try (OutputStream os = connection.getOutputStream();){
                os.write(this.maybeEncodeData(this.serializeBatch(batch)));
            }
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));){
                output = reader.lines().collect(Collectors.joining("\n")).trim();
                code = connection.getResponseCode();
            }
            if (code == 401) {
                log.warn("Received HTTP status 401 from [{}]. Please check your credentials.", (Object)((ClarityEmitter)ClarityEmitter.this).config.baseUrl);
            } else if (code == 413) {
                log.warn("Received HTTP status 413 from [{}]. Batch size of [{}] may be too large, try decreasing.", (Object)((ClarityEmitter)ClarityEmitter.this).config.baseUrl, (Object)((ClarityEmitter)ClarityEmitter.this).config.maxBatchSize);
            } else if (code == 429) {
                if (ClarityEmitter.this.consecutiveTooManyRequests.incrementAndGet() < 12L) {
                    log.info("Received HTTP status 429 from [{}] with message [{}], trying again soon.", (Object)((ClarityEmitter)ClarityEmitter.this).config.baseUrl, (Object)output);
                } else {
                    log.warn("Received HTTP status 429 from [{}] with message [{}], trying again soon.", (Object)((ClarityEmitter)ClarityEmitter.this).config.baseUrl, (Object)output);
                }
            } else if (code / 100 != 2) {
                log.warn("Received HTTP status {} from [{}] with message [{}].", new Object[]{code, ((ClarityEmitter)ClarityEmitter.this).config.baseUrl, output});
            } else {
                ClarityEmitter.this.consecutiveTooManyRequests.set(0L);
                return true;
            }
            return false;
        }

        private Optional<LinkedList<LinkedList<byte[]>>> splitIntoBatches(LinkedList<byte[]> data) {
            long currentLength = 0L;
            LinkedList batches = new LinkedList();
            LinkedList<byte[]> batch = new LinkedList<byte[]>();
            while (!data.isEmpty()) {
                if ((long)data.peek().length + currentLength > (long)((ClarityEmitter)ClarityEmitter.this).config.maxBatchSize) {
                    currentLength = 0L;
                    batches.offer(batch);
                    batch = new LinkedList();
                }
                byte[] item = data.poll();
                batch.offer(item);
                currentLength += (long)item.length;
            }
            if (!batch.isEmpty()) {
                batches.add(batch);
            }
            if (!batches.isEmpty()) {
                return Optional.of(batches);
            }
            return Optional.empty();
        }

        private byte[] serializeBatch(LinkedList<byte[]> batch) {
            Object object;
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            Throwable throwable = null;
            try {
                boolean first = true;
                output.write(91);
                for (byte[] event : batch) {
                    if (first) {
                        first = false;
                    } else {
                        output.write(44);
                    }
                    output.write(event);
                }
                output.write(93);
                object = output.toByteArray();
            }
            catch (Throwable throwable2) {
                try {
                    try {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    catch (Throwable throwable3) {
                        EmittingRunnable.$closeResource(throwable, output);
                        throw throwable3;
                    }
                }
                catch (IOException e) {
                    throw new EmitterException(e);
                }
            }
            EmittingRunnable.$closeResource(throwable, output);
            return object;
        }

        private byte[] maybeEncodeData(byte[] data) throws IOException {
            if (ClarityEmitterConfig.Compression.NONE.equals((Object)((ClarityEmitter)ClarityEmitter.this).config.compression)) {
                return data;
            }
            if (ClarityEmitterConfig.Compression.GZIP.equals((Object)((ClarityEmitter)ClarityEmitter.this).config.compression)) {
                ByteArrayOutputStream bytes = new ByteArrayOutputStream();
                try (GZIPOutputStream output = new GZIPOutputStream(bytes);){
                    output.write(data);
                }
                return bytes.toByteArray();
            }
            throw new EmitterException("Unknown compression type");
        }
    }
}

