import os
import re
import subprocess
import sys
import json
import urllib.parse
import logging

from . import fs
from . import util

def onprem_manager_fe_service_name():
  return "onprem-manager-fe"

def onprem_manager_service_name():
  return "onprem-manager"

def grove_server_service_name():
  return 'grove-server'

def sensu_service_name():
  return 'sensu-client'

def agent_service_name():
  return 'grove-agent'

def always_running_services():
  return [agent_service_name(), sensu_service_name(), grove_server_service_name(), onprem_manager_service_name(),
          onprem_manager_fe_service_name()]


async def write_svdir(svstagedir, svdir, services, syslog=None, user=None):
  """Follow our dreams and become a particular set of services.

  Acquires a lock while doing its work. Returns whether or not any service definition was modified.

  svstagedir - staging service directory
  svdir    - active service directory
  services - dict, service name -> service command
  """

  async with fs.flock(fs.dir_lock(svstagedir)):
    modified = False
    fs.mkdirs(svdir)

    # Unlink and stop services we're not using anymore.
    for service in os.listdir(svdir):
      if __is_managed(svstagedir, svdir, service) and not service.startswith(
          ".") and not service in always_running_services() and not service in services:

        unlink_service(svstagedir, svdir, service)

        # Try to stop immediately. If this fails, continue. runsvdir should stop it within a few seconds.
        try:
          await util.subprocess_call(["sv", "stop", os.path.join(svstagedir, service)])
        except subprocess.CalledProcessError:
          sys.stderr.write("warning: failed to stop service: " + service + "\n")

    # Update and symlink wanted services
    for service, service_defn in services.items():
      modified = await write_service(svstagedir, svdir, service, service_defn, syslog, user) or modified
      link_service(svstagedir, svdir, service)

    return modified

async def bounce(svstagedir, svdir, exclude=[], timeout=0):
  """Bounce all of our currently running services. If any of them fail to bounce, emit a warning and move on.

  Acquires a lock while doing its work.

  svstagedir - staging service directory
  svdir      - active service directory
  exclude    - do not bounce these services
  timeout    - timeout (seconds)
  """

  async with fs.flock(fs.dir_lock(svstagedir)):
    for service in os.listdir(svdir):
      if __is_managed(svstagedir, svdir, service) and service not in exclude:
        await __bounce_one_nolock(svstagedir, svdir, service, timeout)

async def bounce_one(svstagedir, svdir, service, timeout=0):
  """Bounce one currently running service. If it fails to bounce, emit a warning and move on.

  Acquires a lock while doing its work.

  svstagedir - staging service directory
  svdir      - active service directory
  service    - name of the service
  timeout    - timeout (seconds)
  """

  async with fs.flock(fs.dir_lock(svstagedir)):
    await __bounce_one_nolock(svstagedir, svdir, service, timeout)

async def __bounce_one_nolock(svstagedir, svdir, service, timeout=0):
  """Bounce one currently running service. If it fails to bounce, emit a warning and move on.

  Does not acquire a lock. This must be called by some other function that _does_ acquire a lock.

  svstagedir - staging service directory
  svdir      - active service directory
  service    - name of the service
  timeout    - timeout (seconds)
  """

  # Bounce can take a while to work due to runsvdir taking time to start up runsv.
  wait_time = timeout if timeout else 60

  try:
    await util.subprocess_call(["sv", "-w", str(wait_time), "force-reload", os.path.join(svstagedir, service)],
                               wait_time + 10, True)
  except subprocess.CalledProcessError:
    sys.stderr.write("Failed to bounce service: " + service + "\n")

async def stop(svstagedir, svdir, exclude=[], timeout=0):
  """Stop all of our currently running services. If any of them fail to stop, throw an exception.

  Acquires a lock while doing its work.

  svstagedir - staging service directory
  svdir      - active service directory
  exclude    - do not stop these services
  timeout    - timeout (seconds)
  """

  # Stop can take a while to work for a couple of reasons:
  #  (1) The service might not exit immediately.
  #  (2) Maybe we got over-excited and called stop before runsv even started.
  wait_time = timeout if timeout else 60

  async with fs.flock(fs.dir_lock(svstagedir)):
    for service in os.listdir(svdir):
      if __is_managed(svstagedir, svdir, service) and service not in exclude:
        try:
          await util.subprocess_call(["sv", "-w", "15", "stop", os.path.join(svstagedir, service)], wait_time, True)
        except:
          sys.stderr.write("Sending kill signal to service: " + os.path.join(svstagedir, service) + "\n")
          await util.subprocess_call(["sv", "kill", os.path.join(svstagedir, service)], wait_time, True)

async def start(svstagedir, svdir, exclude=[], timeout=0):
  """Start all of our currently running services. If any of them fail to start, throw an exception.

  Acquires a lock while doing its work.

  svstagedir - staging service directory
  svdir      - active service directory
  exclude    - do not start these services
  timeout    - timeout (seconds)
  """

  # Start can take a while to work due to runsvdir taking time to start up runsv.
  wait_time = timeout if timeout else 30

  async with fs.flock(fs.dir_lock(svstagedir)):
    for service in os.listdir(svdir):
      if __is_managed(svstagedir, svdir, service) and service not in exclude:
        await util.subprocess_call(["sv", "start", os.path.join(svstagedir, service)], wait_time, True)

async def status(svstagedir, svdir):
  """Get status of all of our currently running services.

  Acquires a lock while doing its work. Returns results as strings; one per service.

  svstagedir - staging service directory
  svdir    - active service directory
  """
  async with fs.flock(fs.dir_lock(svstagedir)):
    strings = []
    for service in os.listdir(svdir):
      if __is_managed(svstagedir, svdir, service):
        string = util.backtick(["sv", "stat", os.path.join(svstagedir, service)])
        strings.append(string.rstrip())
    return strings

async def _stat(svsstagedir, service):
  """Get status of a currently running service

  Does not acquire a lock. This must be called by some other function that
  _does_ acquire a lock.

  Reads human readable status from {service}/supervise/stat file.  This
  status represents the desired state of the service

  svsstagedir - staging service directory
  service - name of the service
  """
  fstat = os.path.join(svsstagedir, service, "supervise", "stat")
  return fs.read_file(fstat).strip()

async def health(svstagedir, svdir, services, ssl_context=None):
  """Probe health of all managed services

  Acquire a lock while doing its work.

  svstagedir - staging service directory
  svdir - active service directory
  services - list of service names to probe health
  """
  from aiohttp import ClientConnectorError
  logger = logging.getLogger(__name__)

  try:
    async with fs.flock(fs.dir_lock(svstagedir), timeout=0):
      for service in services:
        service_staged = os.path.join(svstagedir, service)
        probe_fname = os.path.join(service_staged, "probe")

        # ensure service exists
        if not os.path.exists(service_staged):
          raise RuntimeError(f"service [{service}] does not exist")

        # ensure service is managed by grove
        if not __is_managed(svstagedir, svdir, service):
          raise RuntimeError(f"service [{service}] not managed by grove")

        # ensure service has a health probe definition
        if not os.path.exists(probe_fname):
          raise RuntimeError(f"service [{service}] has no health probe defined")

        # ensure service is in the "run" state
        state = await _stat(svstagedir, service)
        if state != "run":
          logger.info(f"service [{service}] in desired state [{state}], skipping check")
          continue

        # load health probe definition
        probe_defn = json.loads(fs.read_file(probe_fname))

        try:
          http_probe_defns = probe_defn["http"]

          if not isinstance(http_probe_defns, list):
            raise ValueError(f"service [{service}] health probe definition is invalid")

          for http_probe_defn in http_probe_defns:
            scheme = http_probe_defn.get("scheme", "http")
            host = http_probe_defn.get("host", "127.0.0.1")
            path = http_probe_defn["path"]
            port = http_probe_defn["port"]

            url = urllib.parse.urlunparse((scheme, f"{host}:{port}", path, "", "", ""))
            await util.check_http_get_ok(url, ssl_context=ssl_context, timeout=5)

            logger.debug(f"service [{service}] at [{url}] is healthy")
        except RuntimeError as e:
          raise RuntimeError(f"service [{service}] is not healthy: {e}")
        except ClientConnectorError as e:
          raise RuntimeError(f"service [{service}] is unavailable: {e}")
        except KeyError as e:
          raise ValueError(f"service [{service}] health probe definition requires key [{e}]")
        except Exception:
          raise
  # raised if lock cannot be immediately acquired
  except TimeoutError:
    logger.warning("service health unavailable, grove is busy")
    return True
  except Exception as e:
    logger.debug(e, exc_info=True)
    return False

  return True

def unlink_service(svstagedir, svdir, service):
  """Unlink a service from the svdir. Returns whether the svdir was modified."""
  modified = False
  service_link = os.path.join(svdir, service)
  service_staged = os.path.join(svstagedir, service)

  # Sanity checks:
  if not os.path.islink(service_link):
    raise Exception("service is not a link: " + service)
  elif os.readlink(service_link) != service_staged:
    raise Exception("service is not linked to svstagedir: " + service)
  else:
    modified = True
    os.unlink(service_link)

  return modified

async def write_service(svstagedir, svdir, service, service_defn, syslog=None, user=None):
  """Writes a service into the svstagedir. Returns whether or not the stagedir was modified."""
  modified = False
  command = service_defn["run"]
  health_probe = service_defn.get("healthProbe")
  service_staged = os.path.join(svstagedir, service)

  if not os.path.exists(service_staged):
    modified = True
    fs.mkdirs(os.path.join(service_staged, "supervise"))
    fs.mkdirs(os.path.join(service_staged, "log/main"))
    fs.mkdirs(os.path.join(service_staged, "log/supervise"))
    await util.subprocess_call(['mkfifo', '-m', '0622', os.path.join(service_staged, "supervise/ok")])
    await util.subprocess_call(['mkfifo', '-m', '0622', os.path.join(service_staged, "log/supervise/ok")])

  command_escaped = ""
  for part in command:
    if command_escaped != "":
      command_escaped = command_escaped + " "
    command_escaped = command_escaped + re.escape(part)

  prerun = ""
  if user and service not in always_running_services() and "--root" not in command:
    prerun = f"$([[ \"$USER\" != {user} ]] && echo \"chpst -u {user}\")"

  run_contents = "#!/bin/bash -eu\nexec 2>&1\nexec {} grove-run {}\n".format(prerun, command_escaped)

  log_contents = "#!/bin/bash -eu\nsvlogd ./main\n"

  modified = fs.write_file(os.path.join(service_staged, "run"), run_contents, 0o755) or modified
  modified = fs.write_file(os.path.join(service_staged, "log/run"), log_contents, 0o755) or modified

  if syslog and service in syslog.split(","):
    modified = fs.write_file(os.path.join(service_staged, "log/main/config"), "p[{}] \nu127.0.0.1".format(service),
                             0o644) or modified

  if health_probe is not None and service not in always_running_services():
    modified = fs.write_file(os.path.join(service_staged, "probe"), json.dumps(health_probe)) or modified

  return modified

def link_service(svstagedir, svdir, service):
  """Link a service from the svstagedir into the svdir."""
  service_link = os.path.join(svdir, service)
  service_staged = os.path.join(svstagedir, service)

  if os.path.lexists(service_link):
    # Verify symlink target
    if not os.path.islink(service_link):
      raise Exception("service is not a link: " + service)
    elif not os.readlink(service_link):
      raise Exception("service is not linked to svstagedir: " + service)
  else:
    os.symlink(service_staged, service_link)

def __is_managed(svstagedir, svdir, service):
  """Returns true if this service is managed by Grove, indicated by it being a symlink to Grove's svstagedir"""
  service_link = os.path.join(svdir, service)
  service_staged = os.path.join(svstagedir, service)
  return os.path.islink(service_link) and os.readlink(service_link) == service_staged
