import asyncio
import fcntl
import os
import subprocess
import time

from contextlib import asynccontextmanager

from . import util

@asynccontextmanager
async def flock(file_name, timeout=60):
  """Context manager for acquire_lock."""
  lock = await _acquire_lock(file_name, timeout)
  try:
    yield lock
  finally:
    lock.close()

def dir_lock(dir):
  return os.path.join(dir, ".lock")

def read_file(file_name):
  """Read file contents as a string."""
  with open(file_name, 'r') as f:
    return f.read()

def write_file(file_name, contents, mode=None):
  """Write contents to a file atomically, using a temp file.

  Returns whether the file actually changed."""
  write_needed = False
  old_contents = None

  mkdirs(os.path.dirname(file_name))

  if os.path.exists(file_name):
    with open(file_name, 'r') as f:
      old_contents = f.read()

  if old_contents != contents:
    write_needed = True
    tmp_file_name = file_name + util.random_suffix()
    with open(tmp_file_name, 'w') as f:
      f.write(contents)
      if mode is not None:
        os.fchmod(f.fileno(), mode)
    os.rename(tmp_file_name, file_name)

  return write_needed

def mkdirs(directory):
  if os.path.exists(directory):
    return

  subprocess.check_call(["mkdir", "-m", "0755", "-p", directory])

def rmr(directory):
  subprocess.check_call(["rm", "-fr", directory])
  if os.path.exists(directory):
    raise IOError("Failed to remove directory: " + directory)

def which_executable(file_name, alternate_paths=None):
  paths = []

  try:
    paths.append(os.getcwd())
  except:
    pass

  paths.append(os.path.dirname(os.path.abspath(__file__)))

  if alternate_paths is not None:
    if type(alternate_paths) is list:
      paths.extend(alternate_paths)
    else:
      paths.append(alternate_paths)

  paths.extend(os.environ["PATH"].split(os.pathsep))

  for path in paths:
    abs_file_name = os.path.join(path, file_name)
    if os.path.isfile(abs_file_name) and os.access(abs_file_name, os.X_OK):
      return abs_file_name

  return None

async def _acquire_lock(file_name, timeout):
  """Acquire lock (a particular file) and return the locked fd. Close the fd to release the lock."""
  mkdirs(os.path.dirname(file_name))
  fh = open(file_name, 'w')

  start_time = time.time()
  while True:
    try:
      fcntl.flock(fh, fcntl.LOCK_EX | fcntl.LOCK_NB)
      return fh
    except Exception:
      if time.time() > start_time + timeout:
        raise TimeoutError()
      await asyncio.sleep(1)
