import logging
import os
import urllib

import aiofiles
import aiohttp
import boto3

from . import fs
from . import util

async def fetch_files_async(filenames, outdir, repositories, force_refresh=False, timeout=None, ssl_context=None):
  logger = logging.getLogger(__name__)

  """Fetch filenames from S3 or HTTP into the local file cache."""
  if not filenames:
    return []

  if not repositories:
    raise ValueError("No repositories configured")

  outer_last_exception = None
  num_retries = 3
  for attempt in range(num_retries):
    file_paths = []

    try:
      for filename in reversed(filenames):
        local_path = os.path.join(outdir, filename)
        if force_refresh or not os.path.exists(local_path):
          fs.mkdirs(os.path.dirname(local_path))
          tmp_local_path = local_path + util.random_suffix()

          last_exception = None
          for repository in repositories:
            url = urllib.parse.urlparse("{}/{}".format(repository["url"], filename))

            logger.debug("Trying to fetch [%s]", url.geturl())

            try:
              if url.scheme == "s3":  # TODO: boto is synchronous so S3-schemed downloads aren't actually done async!
                s3 = boto3.resource('s3') if repository.get("user") is None or repository.get(
                  "password") is None else boto3.resource('s3', aws_access_key_id=repository.get("user"), aws_secret_access_key=repository.get("password"))

                key = url.path.lstrip("/")
                s3.Object(url.netloc, key).download_file(tmp_local_path)

              elif url.scheme == "http" or url.scheme == "https":
                await fetch_http_async(
                  url.geturl(),
                  tmp_local_path,
                  repository.get("user"),
                  repository.get("password"),
                  timeout=timeout,
                  ssl_context=ssl_context
                )

              else:
                raise ValueError("URL scheme not supported for: {}".format(url.geturl()))

              os.rename(tmp_local_path, local_path)
              file_paths.append(local_path)

              logger.info("Fetched [%s] from repository [%s]", filename, repository["url"])

              last_exception = None
              repositories = _maybe_promote_repository(repository, repositories)
              break

            except Exception as e:
              logger.debug("Failed to fetch [%s] from [%s], trying next repository: %s", filename, repository["url"], e)
              last_exception = e
              continue  # try the next repository

          if last_exception:
            if os.path.exists(tmp_local_path):
              fs.rmr(tmp_local_path)
            if os.path.exists(local_path):
              file_paths.append(local_path)
              logger.info("Could not refresh [%s], returning object from cache", filename)
            else:
              raise last_exception
        else:
          file_paths.append(local_path)
          logger.debug("File already cached [%s]", filename)

      return file_paths
    except Exception as e:
      logger.debug("fetch_files failed, retrying (%d/%d) - last exception: %s", attempt + 1, num_retries, e)
      outer_last_exception = e
  else:
    if outer_last_exception:
      raise outer_last_exception

async def fetch_http_async(url, output_path, username=None, password=None, mode=None, timeout=None, ssl_context=None):
  auth = aiohttp.BasicAuth(login=username, password=password) if username and password else None
  timeout = aiohttp.ClientTimeout(total=timeout)
  async with aiohttp.ClientSession(auth=auth, auto_decompress=False, timeout=timeout) as session:
    async with session.get(url, ssl=ssl_context) as resp:

      if resp.status / 100 != 2:
        raise RuntimeError("Non 2xx response: {}".format(await resp.content.read()))

      async with aiofiles.open(output_path, 'wb') as fd:
        while True:
          chunk = await resp.content.read(1048576)
          if not chunk:
            break
          await fd.write(chunk)

        if mode:
          os.fchmod(fd.fileno(), mode)

    return await resp.release()

def get_repository_list(config):
  repositories = []

  from_configs = os.environ.get('GROVE_REPOSITORIES', config.get("repositories"))
  if from_configs:
    repositories.extend(from_configs)

  s3_bucket = os.environ.get('GROVE_S3_BUCKET', config.get("s3_bucket"))
  s3_prefix = os.environ.get('GROVE_S3_PREFIX', config.get("s3_prefix"))

  if s3_bucket:
    repositories.append({
      "url": "s3://{}/{}".format(s3_bucket, "" if not s3_prefix else s3_prefix.strip("/")),
      "user": config.get("aws_access_key"),
      "password": config.get("aws_secret_key")
    })

  return [_sanitize_url(x) for x in repositories]

def _sanitize_url(repository):
  repository["url"] = repository["url"].strip("/")
  return repository

def _maybe_promote_repository(repository, repositories):
  """If a repository seems promising because we pulled an artifact from it, promote it to the head of the list so that
  it's tried first for the next file.
  """

  if repository == repositories[0]:
    return repositories

  new_repositories = list(repositories)
  new_repositories.remove(repository)
  new_repositories.insert(0, repository)
  return new_repositories
