import asyncio
import logging
import os
import subprocess

from . import deploy
from . import dirs
from . import fs
from . import sensu_util
from . import services
from . import util

async def become(config, service_yaml, service_type, post_deploy_params=None, always_bounce=False, skip_start=False, syslog=None, post_deploy_params_timeout=120):
  """Becomes a particular service. Returns the linked service directory."""
  homedir = config["homedir"]
  async with fs.flock(os.path.join(homedir, ".lock")):
    svstagedir = dirs.svstagedir(homedir)
    svdir = config["svdir"]
    did_activate = False

    deploy.setup(config)
    pulled = await deploy.pull(config, service_yaml, service_type)

    # Update history file if the pull succeeded.
    deploy.append_history_and_prune_old_files(config, pulled)

    if not deploy.is_activated(homedir, pulled):
      did_activate = True
      # Exclude services marked as always running (Grove agent, Grove server, Sensu, Imply Onprem manager)
      await services.stop(svstagedir, svdir, services.always_running_services())
      deploy.activate(homedir, pulled)

    # Possibly run user-specified commands after symlinking the bundle to the deploy directory but before updating runit's svdir.
    # These can be passed in either as a parameter in the grove-become command or in the deployment YAML.
    await run_post_deploy_commands(service_yaml, post_deploy_params, pulled, post_deploy_params_timeout)

    service_list = mk_service_list(service_yaml["services"], config, service_type)

    # Possibly update runit services.
    did_become = await services.write_svdir(
      svstagedir,
      svdir,
      service_list,
      syslog,
      service_yaml.get("user", config.get("user"))
    )

    if always_bounce or (did_become and not did_activate):
      # Something changed with the services but we didn't activate (so we hadn't stopped). Need to bounce.
      # Exclude services marked as always running (Grove agent, Grove server, Sensu, Imply Onprem manager)
      await services.bounce(svstagedir, svdir, services.always_running_services())

    if services.sensu_service_name() in service_list:
      # Write Sensu client configuration file
      sensu_util.write_config(config, service_yaml["name"], service_type)

      # Always bounce sensu (config may have changed)
      await services.bounce_one(svstagedir, svdir, services.sensu_service_name())

    if not skip_start:
      await services.start(svstagedir, svdir)

    return pulled

async def become_prior(config, hashed_service_parts, post_deploy_params=None, always_bounce=False, skip_start=False, syslog=None, post_deploy_params_timeout=120):
  """Become something that we used to be, given a hashed_service_parts string."""
  homedir = config["homedir"]
  service_yaml, service_type = deploy.read_deploy_json(
    os.path.join(dirs.deploystagedir(homedir), hashed_service_parts, "deploy.json")
  )
  return await become(config, service_yaml, service_type, post_deploy_params, always_bounce, skip_start, syslog, post_deploy_params_timeout)

async def init(config):
  """Sets up grove on this machine and ensures that the svdir is up to date. If grove is not
  already set up, this will also become the nil service. Otherwise, no services will be
  restarted."""
  homedir = config["homedir"]
  async with fs.flock(os.path.join(homedir, ".lock")):
    if not deploy.activated_hash(homedir):
      # Become the nil service.
      pulled = await deploy.pull(config, {'name': 'nil', 'deploy': [], 'env': {}, 'services': {}}, 'nil')

      # Update history file if the pull succeeded.
      deploy.append_history_and_prune_old_files(config, pulled)

      # Activate the nil service.
      deploy.activate(homedir, pulled)

    # Possibly update runit services. Don't restart anything.
    service_yaml, service_type = deploy.read_deploy_json(dirs.deployjson(homedir))
    await services.write_svdir(
      dirs.svstagedir(homedir),
      config["svdir"],
      mk_service_list(service_yaml["services"], config, service_type),
      config.get("syslog"),
      service_yaml.get("user")
    )

def attach_bootstrapped_services(service_list, config):
  if "server_host" in config:
    service_list[services.agent_service_name()] = {'run': ['--root', '--no-environment', 'grove-agent']}

  if "grove_server_dir" in config:
    service_list[services.grove_server_service_name()] = {
      'run': ['--root', '--no-environment', '--', 'grove-server', '--directory', config.get("grove_server_dir")]}

  # When Grove is used to orchestrate an Imply OnPrem cluster, Grove will supervise both the Imply services as well as
  # the manager itself. We don't want the manager to be affected by actions (start, stop, bounce, become, etc.) intended
  # for the services the manager is deploying using Grove.
  if "onprem_manager_path" in config:
    service_list[services.onprem_manager_service_name()] = {
      'run': ['--', config.get("onprem_manager_path"), "manager"]}
  if "onprem_manager_fe_path" in config:
    service_list[services.onprem_manager_fe_service_name()] = {
      'run': ['--', config.get("onprem_manager_fe_path"), '-c',
              config.get("onprem_manager_fe_config_path", "config.yaml")]}

  sensu_path = '/opt/sensu/bin/sensu-client'
  if os.path.exists(sensu_path):
    service_list[services.sensu_service_name()] = {
      'run': ['--', sensu_path, '-c', '/etc/sensu/config.json', '-d', '/etc/sensu/conf.d']}

def mk_service_list(base_services, config, service_type):
  service_list = {}

  if config.get('bootstrap') is not False:
    attach_bootstrapped_services(service_list, config)

  for name, defn in base_services.items():
    if name in service_list:
      raise Exception('Cannot have two services named: {}'.format(name))

    if isinstance(defn, dict) and 'typeFilter' in defn and defn["typeFilter"]:
      type_filters = [defn["typeFilter"]] if isinstance(defn["typeFilter"], str) else defn["typeFilter"]
      for type_prefix in type_filters:
        if type_prefix == service_type or service_type.startswith(type_prefix + "/"):
          service_list[name] = defn
          break
    else:
      service_list[name] = defn

  return service_list

async def run_post_deploy_commands(service_yaml, post_deploy_params, deploy_path, post_deploy_params_timeout):
  logger = logging.getLogger(__name__)

  post_deploy_commands = []
  if post_deploy_params is not None:
    post_deploy_commands.extend(post_deploy_params)

  if "post_deploy_commands" in service_yaml:
    post_deploy_commands.extend(service_yaml["post_deploy_commands"])

  my_env = os.environ.copy()
  my_env["GROVE_DEPLOY"] = deploy_path

  for command in post_deploy_commands:
    logger.info("<Executing: %s>", util.sanitizedLog(command))

    try:
      await util.subprocess_call(command, post_deploy_params_timeout, True, env=my_env, cwd="/")
    except subprocess.CalledProcessError:
      logger.exception(f"Error while executing post deploy command [{command}]")
    except asyncio.TimeoutError:
      logger.exception(f"Timeout exceeded for post deploy command [{command}]")
